// ページ読み込み時にデバッグ情報を出力
console.log('Content script loaded at:', window.location.href);

// デバッグ用の関数
/*
function debugElements() {
  console.log('Running debugElements');
  
  // すべてのtextarea要素を検索
  const allTextareas = document.querySelectorAll('textarea');

  console.log('All textareas found:', allTextareas.length);
  allTextareas.forEach((el, index) => {
    console.log(`Textarea ${index}:`, {
      id: el.id,
      class: el.className,
      attributes: Array.from(el.attributes).map(attr => `${attr.name}="${attr.value}"`).join(', '),
      isVisible: el.offsetParent !== null,
      value: el.value,
      style: el.style.cssText
    });
  });


  // すべてのcontenteditable要素を検索
  const allContentEditables = document.querySelectorAll('[contenteditable="true"]');
  console.log('All contenteditable elements found:', allContentEditables.length);
  allContentEditables.forEach((el, index) => {
    console.log(`Contenteditable ${index}:`, {
      id: el.id,
      class: el.className,
      attributes: Array.from(el.attributes).map(attr => `${attr.name}="${attr.value}"`).join(', '),
      isVisible: el.offsetParent !== null,
      content: el.textContent,
      style: el.style.cssText
    });
  });

  // プロンプト入力に関連しそうな要素を検索
  const possibleInputs = [
    ...allTextareas,
    ...allContentEditables,
    ...document.querySelectorAll('[role="textbox"]'),
    ...document.querySelectorAll('.ProseMirror'),
    ...document.querySelectorAll('#prompt-textarea')
  ];

  /*
  console.log('All possible input elements:', possibleInputs.length);
  possibleInputs.forEach((el, index) => {
    console.log(`Possible input ${index}:`, {
      tagName: el.tagName,
      id: el.id,
      class: el.className,
      attributes: Array.from(el.attributes).map(attr => `${attr.name}="${attr.value}"`).join(', '),
      isVisible: el.offsetParent !== null,
      value: el.value || el.textContent,
      style: el.style.cssText,
      rect: el.getBoundingClientRect()
    });
  });

  return {
    url: window.location.href,
    textareas: Array.from(allTextareas).map(el => ({
      id: el.id,
      class: el.className,
      attributes: Array.from(el.attributes).map(attr => `${attr.name}="${attr.value}"`).join(', '),
      isVisible: el.offsetParent !== null,
      value: el.value,
      style: el.style.cssText
    })),
    contentEditables: Array.from(allContentEditables).map(el => ({
      id: el.id,
      class: el.className,
      attributes: Array.from(el.attributes).map(attr => `${attr.name}="${attr.value}"`).join(', '),
      isVisible: el.offsetParent !== null,
      content: el.textContent,
      style: el.style.cssText
    }))
  };
}
    */

// ページ読み込み完了時にデバッグ情報を出力
window.addEventListener('load', () => {
  console.log('Page fully loaded');
  // debugElements();なんで
});

// 定期的にデバッグ情報を更新（頻度を下げる）
// setInterval(debugElements, 5000);

// プロンプト入力フィールドを探す関数
function findPromptInput() {
  const selectors = [
    '#prompt-textarea',
    'textarea[data-id="root"]',
    'textarea[placeholder*="Send a message"]',
    'textarea[placeholder*="メッセージを送信"]',
    'textarea[placeholder*="質問してみましょう"]',
    'textarea[data-virtualkeyboard="true"]',
    '.ProseMirror[contenteditable="true"]',
    'div[contenteditable="true"]'
  ];

  for (const selector of selectors) {
    const element = document.querySelector(selector);
    if (element) {
      if (element.style.display === 'none') {
        element.style.display = 'block';
      }
      return element;
    }
  }

  return null;
}

// テキストを挿入する関数
function insertText(text) {
  const input = findPromptInput();
  if (!input) {
    return false;
  }

  // 改行コードを統一（\r\n や \r を \n に変換）
  text = text.replace(/\r\n/g, '\n').replace(/\r/g, '\n');
  
  // 先頭と末尾の空行を削除（先頭と末尾の連続する改行を削除）
  text = text.replace(/^\n+/, '').replace(/\n+$/, '');

  if (input.tagName === 'TEXTAREA') {
    input.value = text;
    input.dispatchEvent(new Event('input', { bubbles: true }));
  } else if (input.classList.contains('ProseMirror')) {
    // ProseMirrorエディタの場合
    const lines = text.split('\n');
    
    // 既存の内容をクリア
    input.innerHTML = '';
    
    // 各行を個別のp要素として追加（空行はスキップ）
    lines.forEach((line, index) => {
      // 空行の場合は空のp要素を作成（改行を保持）
      const p = document.createElement('p');
      p.textContent = line;
      input.appendChild(p);
      
      // 最後の行以外は改行を追加
/*      if (index < lines.length - 1) {
        const br = document.createElement('br');
        input.appendChild(br);
      }*/
    });
    
    // 入力イベントを発火
    input.dispatchEvent(new Event('input', { bubbles: true }));
  }
  return true;
}

// クリップボードから貼り付ける関数
async function pasteFromClipboard() {
  const inputElement = findPromptInput();
  if (!inputElement) {
    return { success: false, error: 'No input element found' };
  }

  try {
    inputElement.style.display = 'block';
    inputElement.style.visibility = 'visible';
    inputElement.style.opacity = '1';
    
    inputElement.focus();
    
    // クリップボードから読み込み
    const text = await navigator.clipboard.readText();
    
    // テキストを挿入
    insertText(text);
    
    let parent = inputElement.parentElement;
    while (parent) {
      if (parent.style.display === 'none') {
        parent.style.display = 'block';
      }
      parent = parent.parentElement;
    }
    
    return { success: true };
  } catch (error) {
    return { success: false, error: error.message };
  }
}

// メッセージリスナー
chrome.runtime.onMessage.addListener((request, sender, sendResponse) => {
  if (request.action === 'insertPrompt') {
    const inputElement = findPromptInput();
    if (!inputElement) {
      sendResponse({ success: false, error: 'No input element found' });
      return true;
    }

    // クリップボードを使用する場合の処理
    const processText = async () => {
      try {
        let finalText = request.text;
        
        inputElement.style.display = 'block';
        inputElement.style.visibility = 'visible';
        inputElement.style.opacity = '1';
        
        inputElement.focus();
        
        // クリップボードを使用する場合の処理
        console.log('Processing text in content script:', {
          hasClipPlaceholder: finalText.includes('{{clip}}'),
          hasClipboardContent: !!request.clipboardContent,
          needsClipboard: request.needsClipboard
        });
        
        if (finalText.includes('{{clip}}')) {
          let clipboardText = '';
          
          // popupで読み取れた場合はそれを使用
          if (request.clipboardContent) {
            clipboardText = request.clipboardContent;
            console.log('Using clipboard content from popup, length:', clipboardText.length);
          } else if (request.needsClipboard) {
            console.log('Attempting to read clipboard in content script using execCommand');
            // popupで失敗した場合、入力フィールドに直接貼り付けて取得を試す
            // 入力フィールドにフォーカスしてからexecCommand('paste')を実行
            try {
              // 入力フィールドをクリアしてから貼り付けを試す
              const originalValue = inputElement.tagName === 'TEXTAREA' || inputElement.tagName === 'INPUT' 
                ? inputElement.value 
                : inputElement.textContent || inputElement.innerText;
              
              // 入力フィールドをクリア
              if (inputElement.tagName === 'TEXTAREA' || inputElement.tagName === 'INPUT') {
                inputElement.value = '';
              } else {
                inputElement.textContent = '';
              }
              
              // フォーカスして貼り付けを実行
              inputElement.focus();
              if (inputElement.tagName === 'TEXTAREA' || inputElement.tagName === 'INPUT') {
                inputElement.select();
              }
              
              // 貼り付けを実行
              const pasteSuccess = document.execCommand('paste');
              console.log('execCommand paste result:', pasteSuccess);
              if (pasteSuccess) {
                // 貼り付けられた内容を取得
                if (inputElement.tagName === 'TEXTAREA' || inputElement.tagName === 'INPUT') {
                  clipboardText = inputElement.value;
                  console.log('Clipboard text read from textarea/input, length:', clipboardText.length);
                  inputElement.value = ''; // クリア
                } else {
                  clipboardText = inputElement.textContent || inputElement.innerText;
                  console.log('Clipboard text read from contenteditable, length:', clipboardText.length);
                  inputElement.textContent = ''; // クリア
                }
              } else {
                console.warn('execCommand paste returned false');
                // 貼り付けに失敗した場合は元の値に戻す
                if (inputElement.tagName === 'TEXTAREA' || inputElement.tagName === 'INPUT') {
                  inputElement.value = originalValue;
                } else {
                  inputElement.textContent = originalValue;
                }
              }
            } catch (execError) {
              // execCommandが失敗した場合は空文字で続行
              console.warn('execCommand paste failed, {{clip}} will be replaced with empty string:', execError);
            }
          }
          
          // クリップボードテキストの前後の改行と空白を削除
          clipboardText = clipboardText.trim();
          
          // {{clip}}をクリップボードの内容に置き換え（取得できなかった場合は空文字）
          const beforeReplace = finalText;
          finalText = finalText.replace(/\{\{clip\}\}/g, clipboardText);
          console.log('Text replacement:', {
            before: beforeReplace.substring(0, 100),
            after: finalText.substring(0, 100),
            clipboardTextLength: clipboardText.length
          });
        }
        
        insertText(finalText);
        
        let parent = inputElement.parentElement;
        while (parent) {
          if (parent.style.display === 'none') {
            parent.style.display = 'block';
          }
          parent = parent.parentElement;
        }
        
        sendResponse({ success: true });
      } catch (error) {
        sendResponse({ success: false, error: error.message });
      }
    };
    
    processText();
    return true; // 非同期処理のため true を返す
  } else if (request.action === 'pasteFromClipboard') {
    pasteFromClipboard().then(result => {
      sendResponse(result);
    }).catch(error => {
      sendResponse({ success: false, error: error.message });
    });
    return true; // 非同期処理のため true を返す
  }
  
  return true;
}); 