// 保存されたプロンプトを管理するクラス
class PromptManager {
  constructor() {
    this.prompts = [];
    this.loadPrompts();
    this.draggedItem = null;
  }

  // プロンプトを保存
  savePrompt(title, text, useClipboard = false, id = null) {
    if (!text.trim()) return;
    
    if (id) {
      // 既存のプロンプトを更新
      const index = this.prompts.findIndex(p => p.id === parseInt(id));
      if (index !== -1) {
        this.prompts[index] = {
          ...this.prompts[index],
          title: title.trim() || '無題のプロンプト',
          text: text.trim(),
          useClipboard: useClipboard
        };
      }
    } else {
      // 新しいプロンプトを追加
      const prompt = {
        id: Date.now(),
        title: title.trim() || '無題のプロンプト',
        text: text.trim(),
        useClipboard: useClipboard
      };
      this.prompts.push(prompt);
    }
    
    this.saveToStorage();
    this.updateUI();
  }

  // プロンプトを削除
  deletePrompt(id) {
    this.prompts = this.prompts.filter(p => p.id !== id);
    this.saveToStorage();
    this.updateUI();
  }

  // プロンプトを取得
  getPrompt(id) {
    return this.prompts.find(p => p.id === id);
  }

  // ストレージに保存
  saveToStorage() {
    chrome.storage.local.set({ prompts: this.prompts }, () => {
      console.log('Saved prompts:', this.prompts);
    });
  }

  // ストレージから読み込み
  loadPrompts() {
    chrome.storage.local.get(['prompts'], (result) => {
      console.log('Loaded prompts:', result.prompts);
      this.prompts = result.prompts || [];
      this.updateUI();
    });
  }

  // プロンプトをエクスポート
  exportPrompts() {
    const data = JSON.stringify(this.prompts, null, 2);
    const blob = new Blob([data], { type: 'application/json' });
    const url = URL.createObjectURL(blob);
    const a = document.createElement('a');
    a.href = url;
    a.download = `prompts_${new Date().toISOString().split('T')[0]}.json`;
    document.body.appendChild(a);
    a.click();
    document.body.removeChild(a);
    URL.revokeObjectURL(url);
  }

  // プロンプトをインポート
  importPrompts(file) {
    const reader = new FileReader();
    reader.onload = (e) => {
      try {
        const importedPrompts = JSON.parse(e.target.result);
        if (Array.isArray(importedPrompts)) {
          this.prompts = importedPrompts;
          this.saveToStorage();
          this.updateUI();
        } else {
          alert('無効なファイル形式です。');
        }
      } catch (error) {
        console.error('Error importing prompts:', error);
        alert('ファイルの読み込みに失敗しました。');
      }
    };
    reader.readAsText(file);
  }

  // 編集モードを開始
  startEdit(id) {
    const prompt = this.getPrompt(id);
    if (prompt) {
      document.getElementById('promptTitle').value = prompt.title;
      document.getElementById('promptText').value = prompt.text;
      document.getElementById('useClipboard').checked = prompt.useClipboard || false;
      document.getElementById('saveButton').textContent = '更新';
      document.getElementById('saveButton').dataset.editId = id;
      document.getElementById('cancelButton').style.display = 'block';
    }
  }

  // 編集モードを終了
  endEdit() {
    document.getElementById('promptTitle').value = '';
    document.getElementById('promptText').value = '';
    document.getElementById('useClipboard').checked = false;
    document.getElementById('saveButton').textContent = 'プロンプトを保存';
    delete document.getElementById('saveButton').dataset.editId;
    document.getElementById('cancelButton').style.display = 'none';
  }

  // ドラッグ開始
  handleDragStart(e) {
    this.draggedItem = e.target;
    e.target.classList.add('dragging');
    e.dataTransfer.effectAllowed = 'move';
  }

  // ドラッグ終了
  handleDragEnd(e) {
    e.target.classList.remove('dragging');
    this.draggedItem = null;
  }

  // ドラッグオーバー
  handleDragOver(e) {
    e.preventDefault();
    e.dataTransfer.dropEffect = 'move';
  }

  // ドロップ
  handleDrop(e) {
    e.preventDefault();
    const promptList = document.getElementById('promptList');
    const items = Array.from(promptList.getElementsByClassName('prompt-item'));
    const draggedIndex = items.indexOf(this.draggedItem);
    const droppedIndex = items.indexOf(e.target.closest('.prompt-item'));

    if (draggedIndex !== -1 && droppedIndex !== -1) {
      // プロンプトの順序を更新
      const [movedPrompt] = this.prompts.splice(draggedIndex, 1);
      this.prompts.splice(droppedIndex, 0, movedPrompt);
      this.saveToStorage();
      this.updateUI();
    }
  }

  // UIを更新
  updateUI() {
    console.log('Updating UI with prompts:', this.prompts);
    const promptList = document.getElementById('promptList');
    if (!promptList) {
      console.error('Prompt list element not found');
      return;
    }
    
    promptList.innerHTML = '';

    this.prompts.forEach(prompt => {
      const item = document.createElement('div');
      item.className = 'prompt-item';
      item.draggable = true;
      item.dataset.id = prompt.id;
      
      // ドラッグ&ドロップイベントリスナーを設定
      item.addEventListener('dragstart', (e) => this.handleDragStart(e));
      item.addEventListener('dragend', (e) => this.handleDragEnd(e));
      item.addEventListener('dragover', (e) => this.handleDragOver(e));
      item.addEventListener('drop', (e) => this.handleDrop(e));
      
      const dragHandle = document.createElement('span');
      dragHandle.className = 'drag-handle';
      dragHandle.textContent = '⋮';
      
      const title = document.createElement('span');
      title.textContent = prompt.title;
      
      const useButton = document.createElement('button');
      useButton.className = 'use';
      useButton.textContent = '使用';
      useButton.onclick = async () => {
        try {
          const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });
          
          if (!tab) {
            console.error('No active tab found');
            return;
          }

          if (!tab.url || !tab.url.includes('chatgpt.com')) {
            console.error('Not on ChatGPT page:', tab.url);
            return;
          }

          // クリップボードを使用する場合、popupで読み取る（ユーザージェスチャーのコンテキストを利用）
          let finalText = prompt.text;
          let clipboardContent = '';
          const needsClipboard = prompt.useClipboard && prompt.text.includes('{{clip}}');
          
          console.log('Clipboard processing:', { 
            useClipboard: prompt.useClipboard, 
            hasClipPlaceholder: prompt.text.includes('{{clip}}'),
            needsClipboard: needsClipboard
          });
          
          if (needsClipboard) {
            try {
              // ボタンクリックのユーザージェスチャーのコンテキストでクリップボードを読み取る
              clipboardContent = await navigator.clipboard.readText();
              // 前後の改行と空白を削除
              clipboardContent = clipboardContent.trim();
              console.log('Clipboard read successfully in popup, length:', clipboardContent.length);
              
              // {{clip}}を置き換え
              finalText = finalText.replace(/\{\{clip\}\}/g, clipboardContent);
              console.log('Text after replacement in popup:', finalText.substring(0, 100));
            } catch (clipboardError) {
              // popupでクリップボードを読み取れない場合（セキュリティ制限など）
              console.log('Clipboard read failed in popup, will try in content script');
            }
          }
          
          // テキストを挿入
          console.log('Sending message to content script:', {
            text: finalText.substring(0, 100),
            needsClipboard: needsClipboard && !clipboardContent,
            hasClipboardContent: !!clipboardContent
          });
          
          chrome.tabs.sendMessage(
            tab.id,
            {
              action: 'insertPrompt',
              text: finalText,
              useClipboard: prompt.useClipboard || false,
              needsClipboard: needsClipboard && !clipboardContent,
              clipboardContent: clipboardContent // popupで読み取れた場合は送信
            },
            (response) => {
              if (chrome.runtime.lastError) {
                // content scriptがまだ読み込まれていない場合、読み込んでから再試行
                // これは正常な動作なので、エラーメッセージは表示しない
                chrome.scripting.executeScript({
                  target: { tabId: tab.id },
                  files: ['content.js']
                }).then(() => {
                    setTimeout(() => {
                      chrome.tabs.sendMessage(
                        tab.id,
                        {
                          action: 'insertPrompt',
                          text: finalText,
                          useClipboard: prompt.useClipboard || false,
                          needsClipboard: needsClipboard
                        },
                        (retryResponse) => {
                          if (chrome.runtime.lastError) {
                            // 再試行後も失敗した場合のみエラーを表示
                            console.error('Failed to insert prompt after retry:', chrome.runtime.lastError.message);
                          } else if (retryResponse) {
                            if (retryResponse.success) {
                              window.close();
                            } else if (retryResponse.error) {
                              console.error('Failed to insert prompt:', retryResponse.error);
                            }
                          }
                        }
                      );
                    }, 1000);
                }).catch((error) => {
                  // content scriptの注入に失敗した場合のみエラーを表示
                  console.error('Failed to inject content script:', error);
                });
              } else if (response) {
                if (response.success) {
                  window.close();
                } else if (response.error) {
                  console.error('Failed to insert prompt:', response.error);
                }
              }
            }
          );
        } catch (error) {
          console.error('Error details:', {
            name: error.name,
            message: error.message,
            stack: error.stack
          });
        }
      };

      const editButton = document.createElement('button');
      editButton.className = 'edit';
      editButton.textContent = '編集';
      editButton.onclick = () => this.startEdit(prompt.id);
      
      const deleteButton = document.createElement('button');
      deleteButton.className = 'delete';
      deleteButton.textContent = '削除';
      deleteButton.onclick = () => this.deletePrompt(prompt.id);
      
      item.appendChild(dragHandle);
      item.appendChild(title);
      item.appendChild(useButton);
      item.appendChild(editButton);
      item.appendChild(deleteButton);
      promptList.appendChild(item);
    });
  }
}

// DOMContentLoadedイベントで初期化
document.addEventListener('DOMContentLoaded', () => {
  console.log('DOMContentLoaded event fired');
  
  // プロンプトマネージャーのインスタンスを作成
  const promptManager = new PromptManager();

  // 保存ボタンのイベントリスナー
  const saveButton = document.getElementById('saveButton');
  if (saveButton) {
    console.log('Save button found');
    saveButton.addEventListener('click', () => {
      console.log('Save button clicked');
      const title = document.getElementById('promptTitle').value;
      const promptText = document.getElementById('promptText').value;
      const useClipboard = document.getElementById('useClipboard').checked;
      const editId = saveButton.dataset.editId;
      
      console.log('Saving prompt:', { title, text: promptText, useClipboard, editId });
      promptManager.savePrompt(title, promptText, useClipboard, editId);
      promptManager.endEdit();
    });
  } else {
    console.error('Save button not found');
  }

  // キャンセルボタンのイベントリスナー
  const cancelButton = document.getElementById('cancelButton');
  if (cancelButton) {
    cancelButton.addEventListener('click', () => {
      promptManager.endEdit();
    });
  }

  // エクスポートボタンのイベントリスナー
  const exportButton = document.getElementById('exportButton');
  if (exportButton) {
    exportButton.addEventListener('click', () => {
      promptManager.exportPrompts();
    });
  }

  // インポートボタンのイベントリスナー
  const importButton = document.getElementById('importButton');
  const fileInput = document.getElementById('fileInput');
  if (importButton && fileInput) {
    importButton.addEventListener('click', () => {
      fileInput.click();
    });
    fileInput.addEventListener('change', (e) => {
      if (e.target.files.length > 0) {
        promptManager.importPrompts(e.target.files[0]);
        e.target.value = ''; // 同じファイルを再度選択できるようにする
      }
    });
  }
});

function insertPrompt(text) {
  const textarea = document.querySelector('textarea[data-id="root"]');
  if (textarea) {
    textarea.value = text;
    textarea.dispatchEvent(new Event('input', { bubbles: true }));
  }
} 