// Yahoo検索結果ページから関連キーワードを取得するコンテンツスクリプト
console.log('Yahoo検索結果ページのコンテンツスクリプトが読み込まれました');

// 関連キーワードを取得する関数
function extractRelatedKeywords() {
  console.log('関連キーワードの抽出を開始します');
  
  // 関連キーワードのセレクタ（複数のパターンを試す）
  const selectors = [
    '.SouthUnitItem__text', // 提供されたHTMLのクラス
    '.SouthUnitItem__inner .SouthUnitItem__text',
    '[class*="SouthUnitItem"] .util-Clamps--2',
    '.rel-bot .SouthUnitItem__text',
    'a[data-cl-params*="rel-bot"] .SouthUnitItem__text'
  ];
  
  let relatedKeywords = [];
  
  // 各セレクタを試して関連キーワードを取得
  for (const selector of selectors) {
    const elements = document.querySelectorAll(selector);
    console.log(`セレクタ "${selector}" で ${elements.length} 個の要素が見つかりました`);
    
    elements.forEach((element, index) => {
      const text = element.textContent?.trim();
      if (text && text.length > 0) {
        // 太字タグを除去してテキストのみを取得
        const cleanText = text.replace(/<[^>]*>/g, '').trim();
        if (cleanText && !relatedKeywords.includes(cleanText)) {
          relatedKeywords.push(cleanText);
          console.log(`関連キーワード ${index + 1}: ${cleanText}`);
        }
      }
    });
    
    // 関連キーワードが見つかったらループを抜ける
    if (relatedKeywords.length > 0) {
      break;
    }
  }
  
  // より広範囲な検索（フォールバック）
  if (relatedKeywords.length === 0) {
    console.log('フォールバック検索を実行します');
    
    // ページ内のすべてのリンクをチェック
    const allLinks = document.querySelectorAll('a[href*="/search?p="]');
    console.log(`検索リンクが ${allLinks.length} 個見つかりました`);
    
    allLinks.forEach((link, index) => {
      const text = link.textContent?.trim();
      if (text && text.length > 0 && text.length < 50) { // 適切な長さのテキストのみ
        const cleanText = text.replace(/<[^>]*>/g, '').trim();
        if (cleanText && !relatedKeywords.includes(cleanText)) {
          relatedKeywords.push(cleanText);
          console.log(`フォールバック関連キーワード ${index + 1}: ${cleanText}`);
        }
      }
    });
  }
  
  console.log(`合計 ${relatedKeywords.length} 個の関連キーワードを取得しました:`, relatedKeywords);
  return relatedKeywords;
}

// ページが完全に読み込まれた後に実行
function waitForPageLoad() {
  return new Promise((resolve) => {
    if (document.readyState === 'complete') {
      resolve();
    } else {
      window.addEventListener('load', resolve);
    }
  });
}

// 関連キーワードの取得を実行
async function getRelatedKeywords() {
  try {
    // ページの読み込み完了を待つ
    await waitForPageLoad();
    
    // 少し待ってから関連キーワードを取得（動的コンテンツの読み込みを待つ）
    setTimeout(() => {
      const keywords = extractRelatedKeywords();
      
      if (keywords.length > 0) {
        // 関連キーワードをストレージに保存
        chrome.storage.local.set({ 
          relatedKeywords: keywords,
          lastSearchUrl: window.location.href,
          extractedAt: new Date().toISOString()
        }, () => {
          console.log('関連キーワードをストレージに保存しました:', keywords);
          
          // バックグラウンドスクリプトに通知
          chrome.runtime.sendMessage({
            action: 'relatedKeywordsExtracted',
            keywords: keywords,
            url: window.location.href
          });
        });
      } else {
        console.log('関連キーワードが見つかりませんでした');
      }
    }, 2000); // 2秒待機
    
  } catch (error) {
    console.error('関連キーワードの取得中にエラーが発生しました:', error);
  }
}

// メッセージリスナー
chrome.runtime.onMessage.addListener((request, sender, sendResponse) => {
  if (request.action === 'extractRelatedKeywords') {
    console.log('関連キーワードの抽出を要求されました');
    getRelatedKeywords();
    sendResponse({ success: true });
  }
  
  return true;
});

// ページ読み込み時に自動実行
console.log('ページ読み込み完了、関連キーワードの取得を開始します');
getRelatedKeywords();