// Yahoo検索拡張機能のポップアップスクリプト
console.log('Yahoo検索拡張機能が読み込まれました');

// Chrome拡張機能APIが利用可能かチェック
function isChromeExtensionAPIAvailable() {
  return typeof chrome !== 'undefined' && 
         chrome.storage && 
         chrome.storage.local &&
         chrome.tabs &&
         chrome.runtime;
}

// 安全なストレージ操作のためのラッパー関数
function safeStorageGet(keys, callback) {
  if (!isChromeExtensionAPIAvailable()) {
    console.error('Chrome拡張機能APIが利用できません');
    callback({});
    return;
  }
  
  try {
    chrome.storage.local.get(keys, (result) => {
      if (chrome.runtime.lastError) {
        console.error('ストレージからの読み込みエラー:', chrome.runtime.lastError);
        callback({});
        return;
      }
      callback(result);
    });
  } catch (error) {
    console.error('ストレージAPIの使用中にエラーが発生しました:', error);
    callback({});
  }
}

function safeStorageSet(data, callback) {
  if (!isChromeExtensionAPIAvailable()) {
    console.error('Chrome拡張機能APIが利用できません');
    if (callback) callback();
    return;
  }
  
  try {
    chrome.storage.local.set(data, () => {
      if (chrome.runtime.lastError) {
        console.error('ストレージへの保存エラー:', chrome.runtime.lastError);
        if (callback) callback();
        return;
      }
      if (callback) callback();
    });
  } catch (error) {
    console.error('ストレージAPIの使用中にエラーが発生しました:', error);
    if (callback) callback();
  }
}

// DOM要素を格納する変数
let searchKeywordInput;
let searchButton;
let relatedKeywordsContainer;
let keywordsList;
let copyKeywordsButton;

// 関連キーワード表示を初期化する関数
function clearStoredRelatedKeywords(callback) {
  safeStorageSet(
    { relatedKeywords: [], lastSearchUrl: '', extractedAt: null },
    callback
  );
}

function showProcessingMessage(message = '関連キーワードを取得中...') {
  if (!keywordsList || !relatedKeywordsContainer) {
    return;
  }

  const statusMessage = document.createElement('div');
  statusMessage.className = 'status-message';
  statusMessage.textContent = message;

  keywordsList.innerHTML = '';
  keywordsList.appendChild(statusMessage);
  relatedKeywordsContainer.style.display = 'block';
}

function prepareRelatedKeywordsForNewSearch() {
  showProcessingMessage();
  clearStoredRelatedKeywords();
}

// Yahoo検索を実行する関数
function performYahooSearch(keyword) {
  if (!keyword || keyword.trim() === '') {
    alert('検索キーワードを入力してください');
    return;
  }

  prepareRelatedKeywordsForNewSearch();
  
  // キーワードをURLエンコード
  const encodedKeyword = encodeURIComponent(keyword.trim());
  
  // Yahoo検索URLを構築
  const yahooSearchUrl = `https://search.yahoo.co.jp/search?p=${encodedKeyword}`;
  
  console.log('検索URL:', yahooSearchUrl);
  
  // 新しいタブでYahoo検索結果を開く
  chrome.tabs.create({
    url: yahooSearchUrl,
    active: true
  }, (tab) => {
    if (chrome.runtime.lastError) {
      console.error('タブの作成に失敗しました:', chrome.runtime.lastError);
      alert('検索ページを開けませんでした。もう一度お試しください。');
    } else {
      console.log('検索タブが作成されました:', tab.id);
      
      // 関連キーワードの取得を開始（少し遅延させてページの読み込みを待つ）
      setTimeout(() => {
        chrome.tabs.sendMessage(tab.id, { action: 'extractRelatedKeywords' }, (response) => {
          if (chrome.runtime.lastError) {
            console.log('コンテンツスクリプトからの応答待機中...');
          }
        });
      }, 3000);
      
      // ポップアップを閉じる
//      window.close();
    }
  });
}

// イベントリスナーを設定する関数
function setupEventListeners() {
  // 検索ボタンのクリックイベント
  searchButton.addEventListener('click', () => {
    const keyword = searchKeywordInput.value;
    saveSearchKeywordOnSearch(keyword);
    performYahooSearch(keyword);
  });

  // Enterキーでの検索
  searchKeywordInput.addEventListener('keypress', (event) => {
    if (event.key === 'Enter') {
      const keyword = searchKeywordInput.value;
      saveSearchKeywordOnSearch(keyword);
      performYahooSearch(keyword);
    }
  });
  
  // コピーボタンのクリックイベント
  copyKeywordsButton.addEventListener('click', async () => {
    try {
      // 現在の関連キーワードを取得
      safeStorageGet(['relatedKeywords'], (result) => {
        if (result.relatedKeywords && result.relatedKeywords.length > 0) {
          // 関連キーワードを改行区切りで結合
          const keywordsText = result.relatedKeywords.join('\n');
          
          // クリップボードにコピー
          navigator.clipboard.writeText(keywordsText).then(() => {
            console.log('関連キーワードをコピーしました:', keywordsText);
            
            // ボタンの見た目を一時的に変更
            copyKeywordsButton.textContent = 'コピー済み';
            copyKeywordsButton.classList.add('copied');
            
            // 2秒後に元に戻す
            setTimeout(() => {
              copyKeywordsButton.textContent = 'コピー';
              copyKeywordsButton.classList.remove('copied');
            }, 2000);
            
          }).catch((error) => {
            console.error('クリップボードへのコピーに失敗しました:', error);
            alert('コピーに失敗しました。手動でコピーしてください。');
          });
        } else {
          alert('コピーする関連キーワードがありません。');
        }
      });
    } catch (error) {
      console.error('コピー処理中にエラーが発生しました:', error);
      alert('コピー処理中にエラーが発生しました。');
    }
  });
}

// 関連キーワードを表示する関数
function displayRelatedKeywords(keywords) {
  // DOM要素の存在確認
  if (!keywordsList) {
    console.error('keywordsList要素が見つかりません');
      return;
    }
    
  if (!relatedKeywordsContainer) {
    console.error('relatedKeywordsContainer要素が見つかりません');
            return;
          }

  if (!keywords || keywords.length === 0) {
    keywordsList.innerHTML = '<div class="no-keywords">関連キーワードが見つかりませんでした</div>';
            return;
          }

  keywordsList.innerHTML = '';
  keywords.forEach((keyword, index) => {
    const keywordTag = document.createElement('div');
    keywordTag.className = 'keyword-tag';
    keywordTag.textContent = keyword;
    keywordTag.title = `「${keyword}」で検索`;
    
    // クリックでそのキーワードで検索
    keywordTag.addEventListener('click', () => {
      if (searchKeywordInput) {
        searchKeywordInput.value = keyword;
        performYahooSearch(keyword);
      }
    });
    
    keywordsList.appendChild(keywordTag);
  });
  
  // 関連キーワードエリアを表示
  relatedKeywordsContainer.style.display = 'block';
}

// 関連キーワードを取得する関数
function loadRelatedKeywords() {
  safeStorageGet(['relatedKeywords', 'lastSearchUrl', 'extractedAt'], (result) => {
    if (result.relatedKeywords && result.relatedKeywords.length > 0) {
      console.log('保存された関連キーワードを読み込みました:', result.relatedKeywords);
      displayRelatedKeywords(result.relatedKeywords);
                } else {
      console.log('関連キーワードが見つかりませんでした');
      if (relatedKeywordsContainer) {
        relatedKeywordsContainer.style.display = 'none';
      }
    }
  });
}


// ページ読み込み時の処理
document.addEventListener('DOMContentLoaded', () => {
  console.log('ポップアップが読み込まれました');
  
  // DOM要素を取得
  searchKeywordInput = document.getElementById('searchKeyword');
  searchButton = document.getElementById('searchButton');
  relatedKeywordsContainer = document.getElementById('relatedKeywordsContainer');
  keywordsList = document.getElementById('keywordsList');
  copyKeywordsButton = document.getElementById('copyKeywords');
  
  // DOM要素の存在確認
  if (!searchKeywordInput || !searchButton || !relatedKeywordsContainer || !keywordsList || !copyKeywordsButton) {
    console.error('必要なDOM要素が見つかりません');
    return;
  }
  
  // 入力フィールドにフォーカス
  searchKeywordInput.focus();
  
  // 保存された検索履歴があれば復元（オプション）
  safeStorageGet(['lastSearchKeyword'], (result) => {
    if (result.lastSearchKeyword) {
      searchKeywordInput.value = result.lastSearchKeyword;
    }
  });
  
  // イベントリスナーを設定
  setupEventListeners();
  
  // 関連キーワードを読み込み
  loadRelatedKeywords();
  
  // 関連キーワードの定期チェックを開始
  startKeywordsCheck();
});

// 検索キーワードを保存（オプション）
function saveSearchKeyword(keyword) {
  safeStorageSet({ lastSearchKeyword: keyword }, () => {
    console.log('検索キーワードを保存しました:', keyword);
  });
}

// 検索実行時にキーワードを保存する関数
function saveSearchKeywordOnSearch(keyword) {
  if (keyword && keyword.trim()) {
    saveSearchKeyword(keyword);
  }
}

// 関連キーワードを定期的にチェックする関数
function checkForRelatedKeywords() {
  safeStorageGet(['relatedKeywords', 'extractedAt'], (result) => {
    if (result.relatedKeywords && result.relatedKeywords.length > 0) {
      console.log('関連キーワードをチェックして更新しました:', result.relatedKeywords);
      displayRelatedKeywords(result.relatedKeywords);
      }
    });
  }

// 定期的に関連キーワードをチェック（ポップアップが開いている間のみ）
let keywordsCheckInterval;

// ポップアップが表示されている間は定期的にチェック
function startKeywordsCheck() {
  if (keywordsCheckInterval) {
    clearInterval(keywordsCheckInterval);
  }
  
  keywordsCheckInterval = setInterval(() => {
    checkForRelatedKeywords();
  }, 1000); // 1秒ごとにチェック
}

// ポップアップが閉じる時にチェックを停止
function stopKeywordsCheck() {
  if (keywordsCheckInterval) {
    clearInterval(keywordsCheckInterval);
    keywordsCheckInterval = null;
  }
}

// バックグラウンドスクリプトからのメッセージを受信
chrome.runtime.onMessage.addListener((request, sender, sendResponse) => {
  if (request.action === 'relatedKeywordsUpdated') {
    console.log('バックグラウンドから関連キーワードの更新通知を受信:', request.keywords);
    displayRelatedKeywords(request.keywords);
  }
  
  return true;
});

// ページがアンロードされる時にチェックを停止
window.addEventListener('beforeunload', stopKeywordsCheck);